/* Copyright (c) 2016-2018 VMware, Inc. All rights reserved. */
package com.vmware.automatic.plugin.registration.commands;

import com.vmware.automatic.plugin.registration.commands.utils.PrintHelper;
import com.vmware.automatic.plugin.registration.commands.utils.TrustAllTrustManager;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.ServiceContent;
import com.vmware.vim25.VimPortType;
import com.vmware.vim25.VimService;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSession;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.soap.SOAPFaultException;
import java.util.Map;

/**
 * Base abstract class with base arguments. If need to implement other commands need to extended.
 */
public abstract class PluginCmd {
   private static final ManagedObjectReference SVC_INST_REF = new ManagedObjectReference();
   /**
    * Name and Type of Service instance
    */
   private static final String SVC_INST_NAME = "ServiceInstance";

   protected String _url;
   protected String _username;
   protected String _password;
   protected String _key = "";
   protected CommandLine _commandLine;
   protected ManagedObjectReference _extensionManager;
   protected VimPortType _vimPort;
   protected String _action = "";
   protected String _errorMessage = "";
   private VimService _vimService;
   private ServiceContent _serviceContent;
   private boolean _isConnected = false;

   /**
    * Name and description of command line arguments
    */
   public Options getPluginOpts() {
      Options options = new Options();
      options.addOption(Option.builder("action").hasArg().argName("action")
            .desc("registerPlugin, unregisterPlugin, updatePlugin, isPluginRegistered").required().build());
      options.addOption(Option.builder("url").hasArg().argName("vc url")
            .desc("vCenter server URL").required().build());
      options.addOption(Option.builder("u").longOpt("username").hasArg().argName("vc user")
            .desc("vCenter server username").required().build());
      options.addOption(Option.builder("p").longOpt("password").hasArg().argName("vc pass")
            .desc("vCenter server password").required().build());
      options.addOption(Option.builder("k").longOpt("key").hasArg().argName("key").desc("Plugin key").required()
            .build());
      return options;
   }

   /**
    * Connects to the vCenter, executes the current command and disconnects.
    * @param args
    * @throws Exception
    */
   public void execute(String[] args) {
      populateFields(args);
      try {
         connect();
         doExecute();
      } catch (Exception e) {
         if (e.getMessage() != null) {
            System.out.println(e.getMessage());
         }
      } finally {
         try {
            disconnect();
         } catch (SOAPFaultException sfe) {
            PrintHelper.printSoapFaultException(sfe);
         } catch (Exception e) {
            System.out.println("Failed to disconnect - " + e.getMessage());
            e.printStackTrace();
         }
      }
   }

   /**
    * Populate command line arguments
    */
   protected void populateFields(String[] commandLineArgs) {
      Options options = getPluginOpts();
      CommandLineParser parser = new DefaultParser();
      try {
         _commandLine = parser.parse(options, commandLineArgs);
      } catch (ParseException e) {
         new HelpFormatter().printHelp(
               _action,
               _errorMessage,
               options,
               "\n" + e.getMessage(),
               true);
         return;
      }
      _url = _commandLine.getOptionValue("url");
      _username = _commandLine.getOptionValue("username");
      _password = _commandLine.getOptionValue("password");
      _key = _commandLine.getOptionValue("key");
   }

   /**
    * Establishes user session with the vCenter server.
    *
    * @throws Exception
    */
   private void connect() throws Exception {

      if (!_isConnected) {
         HostnameVerifier hv = new HostnameVerifier() {
            public boolean verify(String urlHostName, SSLSession session) {
               return true;
            }
         };
         TrustAllTrustManager.trustAllHttpsCertificates();
         HttpsURLConnection.setDefaultHostnameVerifier(hv);

         SVC_INST_REF.setType(SVC_INST_NAME);
         SVC_INST_REF.setValue(SVC_INST_NAME);

         _vimService = new VimService();
         _vimPort = _vimService.getVimPort();
         Map<String, Object> ctxt = ((BindingProvider) _vimPort).getRequestContext();

         ctxt.put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, _url);
         ctxt.put(BindingProvider.SESSION_MAINTAIN_PROPERTY, true);

         _serviceContent = _vimPort.retrieveServiceContent(SVC_INST_REF);
         _vimPort.login(_serviceContent.getSessionManager(), _username, _password, null);
         _isConnected = true;

         _extensionManager = _serviceContent.getExtensionManager();
      }
   }

   /**
    * Disconnects the user session.
    *
    * @throws Exception
    */
   private void disconnect() throws Exception {
      if (_isConnected) {
         _vimPort.logout(_serviceContent.getSessionManager());
      }
      _isConnected = false;
   }

   /**
    * Implement actual work for the command.
    */
   protected abstract void doExecute();

}
